// Spotify OAuth (PKCE) + Web API control for MV3 background service worker
// NOTE: Do NOT embed client secret in extensions. We use Authorization Code w/ PKCE.
(function() {
  // Client ID is now set by user in settings
  let CLIENT_ID = null;
  // Prefer the provided fixed redirect if available, else fall back to runtime-based.
  const FIXED_REDIRECT_URI = 'https://jjdoboloblglnjckcdfoofobadndfdlk.chromiumapp.org/callback';
  const RUNTIME_REDIRECT_URI = `https://${chrome.runtime.id}.chromiumapp.org/callback`;
  const REDIRECT_URI = FIXED_REDIRECT_URI || RUNTIME_REDIRECT_URI;

  const AUTH_BASE = 'https://accounts.spotify.com/authorize';
  const TOKEN_URL = 'https://accounts.spotify.com/api/token';
  const API_BASE = 'https://api.spotify.com/v1';
  // Scopes required for remote playback control and reading current state
  const SCOPES = [
    'user-read-playback-state',
    'user-modify-playback-state',
    'user-read-currently-playing'
  ];

  // util: base64url encode (ArrayBuffer -> string)
  function base64UrlEncode(buffer) {
    const bytes = new Uint8Array(buffer);
    let binary = '';
    for (let i = 0; i < bytes.byteLength; i++) binary += String.fromCharCode(bytes[i]);
    return btoa(binary).replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
  }

  // PKCE helpers
  async function createCodeVerifierChallenge() {
    const array = new Uint8Array(32);
    crypto.getRandomValues(array);
    const codeVerifier = base64UrlEncode(array);
    const enc = new TextEncoder().encode(codeVerifier);
    const digest = await crypto.subtle.digest('SHA-256', enc);
    const codeChallenge = base64UrlEncode(digest);
    return { codeVerifier, codeChallenge };
  }

  async function saveSpotifyState(state) {
    const current = (await chrome.storage.local.get('spotify')).spotify || {};
    await chrome.storage.local.set({ spotify: { ...current, ...state } });
  }

  async function getSpotifyState() {
    const data = await chrome.storage.local.get('spotify');
    return data.spotify || {};
  }

  async function getClientId() {
    const state = await getSpotifyState();
    return state.client_id || null;
  }

  async function setClientId(clientId) {
    const current = await getSpotifyState();
    await chrome.storage.local.set({ 
      spotify: { 
        ...current, 
        client_id: clientId 
      } 
    });
    CLIENT_ID = clientId;
  }

  async function startAuthInteractive() {
    CLIENT_ID = await getClientId();
    if (!CLIENT_ID) {
      throw new Error('クライアントIDが設定されていません。設定でSpotifyクライアントIDを入力してください。');
    }
    
    const { codeVerifier, codeChallenge } = await createCodeVerifierChallenge();
    await saveSpotifyState({ codeVerifier });
    const params = new URLSearchParams({
      client_id: CLIENT_ID,
      response_type: 'code',
      redirect_uri: REDIRECT_URI,
      code_challenge_method: 'S256',
      code_challenge: codeChallenge,
      scope: SCOPES.join(' '),
      show_dialog: 'true'
    });
    const authUrl = `${AUTH_BASE}?${params.toString()}`;

    const redirectUrl = await chrome.identity.launchWebAuthFlow({
      url: authUrl,
      interactive: true
    });

    const url = new URL(redirectUrl);
    const code = url.searchParams.get('code');
    if (!code) throw new Error('Missing authorization code');

    return await exchangeCodeForTokens(code);
  }

  async function exchangeCodeForTokens(code) {
    CLIENT_ID = await getClientId();
    if (!CLIENT_ID) {
      throw new Error('クライアントIDが設定されていません');
    }
    
    const { codeVerifier } = await getSpotifyState();
    const body = new URLSearchParams({
      client_id: CLIENT_ID,
      grant_type: 'authorization_code',
      code,
      redirect_uri: REDIRECT_URI,
      code_verifier: codeVerifier
    });

    const resp = await fetch(TOKEN_URL, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body
    });
    if (!resp.ok) throw new Error(`Token exchange failed: ${resp.status}`);
    const json = await resp.json();
    const expiresAt = Date.now() + (json.expires_in * 1000) - 60000; // minus 60s buffer
    await saveSpotifyState({
      access_token: json.access_token,
      refresh_token: json.refresh_token,
      expires_at: expiresAt,
      scopes: SCOPES
    });
    return await getSpotifyState();
  }

  async function refreshAccessToken() {
    CLIENT_ID = await getClientId();
    if (!CLIENT_ID) {
      throw new Error('クライアントIDが設定されていません');
    }
    
    const { refresh_token } = await getSpotifyState();
    if (!refresh_token) throw new Error('No refresh token');

    const body = new URLSearchParams({
      client_id: CLIENT_ID,
      grant_type: 'refresh_token',
      refresh_token
    });
    const resp = await fetch(TOKEN_URL, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body
    });
    if (!resp.ok) throw new Error(`Refresh failed: ${resp.status}`);
    const json = await resp.json();
    const updates = {
      access_token: json.access_token,
      expires_at: Date.now() + (json.expires_in * 1000) - 60000
    };
    if (json.refresh_token) updates.refresh_token = json.refresh_token;
    await saveSpotifyState(updates);
    return await getSpotifyState();
  }

  async function getAccessTokenEnsured() {
    let { access_token, expires_at } = await getSpotifyState();
    if (!access_token || !expires_at || Date.now() >= expires_at) {
      await refreshAccessToken();
      ({ access_token } = await getSpotifyState());
    }
    return access_token;
  }

  async function api(method, path, body) {
    const token = await getAccessTokenEnsured();
    const resp = await fetch(`${API_BASE}${path}`, {
      method,
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: body ? JSON.stringify(body) : undefined
    });
    if (resp.status === 204) return null;
    if (!resp.ok) {
      const text = await resp.text().catch(() => '');
      throw new Error(`Spotify API ${method} ${path} failed: ${resp.status} ${text}`);
    }
    return await resp.json();
  }

  // High-level controls
  async function getPlayer() { return api('GET', '/me/player'); }
  async function play(body) { return api('PUT', '/me/player/play', body); }
  async function pause() { return api('PUT', '/me/player/pause'); }
  async function next() { return api('POST', '/me/player/next'); }
  async function previous() { return api('POST', '/me/player/previous'); }
  async function setVolume(percent) { return api('PUT', `/me/player/volume?volume_percent=${percent}`); }
  async function setShuffle(state) { return api('PUT', `/me/player/shuffle?state=${state}`); }
  async function setRepeat(state) { return api('PUT', `/me/player/repeat?state=${state}`); }
  async function getDevices() { return api('GET', '/me/player/devices'); }
  async function transferPlayback(deviceId, playOnTransfer) { return api('PUT', '/me/player', { device_ids: [deviceId], play: !!playOnTransfer }); }

  // Message bridge for popup/content scripts
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (!message || !message.action || !message.action.startsWith('SPOTIFY_')) return;
    (async () => {
      try {
        switch (message.action) {
          case 'SPOTIFY_STATUS': {
            const s = await getSpotifyState();
            const clientId = await getClientId();
            const authed = !!s.access_token && !!s.refresh_token && !!clientId;
            let player = null;
            if (authed) {
              try { player = await getPlayer(); } catch (e) {/* ignore */}
            }
            sendResponse({ success: true, authed, player, redirect: REDIRECT_URI, hasClientId: !!clientId });
            break;
          }
          case 'SPOTIFY_SET_CLIENT_ID': {
            await setClientId(message.clientId);
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_GET_CLIENT_ID': {
            const clientId = await getClientId();
            sendResponse({ success: true, clientId });
            break;
          }
          case 'SPOTIFY_AUTH': {
            const state = await startAuthInteractive();
            sendResponse({ success: true, state });
            break;
          }
          case 'SPOTIFY_TOGGLE': {
            const player = await getPlayer();
            if (player && player.is_playing) await pause(); else await play();
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_PLAY': {
            await play(message.body);
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_PAUSE': {
            await pause();
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_NEXT': {
            await next();
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_PREV': {
            await previous();
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_VOLUME': {
            await setVolume(message.percent);
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_SHUFFLE': {
            await setShuffle(!!message.state);
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_REPEAT': {
            await setRepeat(message.state || 'off');
            sendResponse({ success: true });
            break;
          }
          case 'SPOTIFY_DEVICES': {
            const d = await getDevices();
            sendResponse({ success: true, devices: d });
            break;
          }
          case 'SPOTIFY_TRANSFER': {
            await transferPlayback(message.deviceId, message.play);
            sendResponse({ success: true });
            break;
          }
          default:
            sendResponse({ success: false, error: 'Unknown action' });
        }
      } catch (e) {
        sendResponse({ success: false, error: e.message });
      }
    })();
    return true; // async
  });
})();

